package jpattern.compiler;

import jpattern.pattern.Pattern;
import jpattern.pattern.PatternCode;

import jpattern.util.Error;
import jpattern.util.Debug;
import jpattern.util.CharStream;
import jpattern.util.QuotedString;
import jpattern.pattern.Variable;

import static jpattern.util.QuotedString.*;
import static jpattern.compiler.Keyword.*;

import java.util.HashMap;
import java.util.EnumSet;
import java.util.StringTokenizer;
import java.util.Stack;

public class Compiler
{
    static final char DQUOTE = QuotedString.DQUOTE;
    static final char SQUOTE = QuotedString.SQUOTE;
    static final char LPAREN = QuotedString.LPAREN;
    static final char RPAREN = QuotedString.RPAREN;

    static final int DEFAULTINDENTINCR = 4;

    static boolean isAltFollow(int ttype)
	{return (ttype==Token.BAR)
		||(ttype==Token.EOF)
		||(ttype==Token.RPAREN)
	        ;
	}

    CharStream cs;   
    Lexer lex;
    HashMap<String,Keyword> Keywords = new HashMap<String,Keyword>();

    int ttype;
    String token;
    char lexquote = DQUOTE;

    public Compiler()
    {
    }

    public void setQuote(char q) throws Error
	{lexquote=q; if(lex!=null) lex.setQuote(q);}

    public Pattern compile(String expr) throws Error
    {
	Parsetree tree = parse(expr);
	Debug.level(2).println("tree="+tree.toString()); 
	Pattern p = compile(tree);
	return p;
    }

    public String compileJ(String expr) throws Error
	{return compileJ(expr,0,DEFAULTINDENTINCR);}

    public String compileJ(String expr, int indent) throws Error
	{return compileJ(expr,indent,DEFAULTINDENTINCR);}

    public String compileJ(String expr,int indent, int incr) throws Error
    {
	Parsetree tree = parse(expr);
	Debug.level(2).println("tree="+tree); 
	compileJ(tree,false); // generate without indents initially
	return indent(tree.text,indent,incr);
    }

    public Parsetree parse(String expr) throws Error
    {
        cs = new CharStream(expr);
	lex = new Lexer(cs);
	lex.setQuote(lexquote);
	// prime pump
	advance();
	Parsetree node = null;
	try {
	    node = parsePat();
	} catch (Exception e) {
	    node = null;
	    int cursor = cs.getCursor();
	    String s = (String)cs.getSequence();
	    String prefix = s.substring(0,cursor);
	    String suffix = s.substring(cursor,s.length());
Debug.level(2).println(e.toString()+"; "+prefix+"|"+suffix);
if(Debug.islevel(2)) e.printStackTrace();
	    throw new Error(prefix+"|"+suffix,e);
	}
	// Check to see that whole expr was consumed
	int cursor = cs.getCursor();
	if(cursor < expr.length()) {
	    throw new Error("Compiler.parse: expr incompletely consumed: |"
			     + expr.substring(cursor,expr.length())+"|");
	}	
	return node;
    }

    public Pattern compile(Parsetree n) throws Error
    {
	Parsetree l;
	Pattern p = null;
	String s = null;

	switch (n.code) {
	case ALT:
	    p = Pattern.Alternate(compile(n.left),compile(n.right));
	    break;
	case ANY:
	    switch ((l=n.left).code) {
	    case _STRING: p = Pattern.Any(l.stringValue()); break;
	    case _VAR: p = Pattern.Any(l.varValue()); break;
	    }
	    break;
	case ARB:
	    p = Pattern.Arb();
	    break;
	case ARBNO:
	    if(n.left.code == _STRING)
		p = Pattern.Arbno(n.left.stringValue());
	    else
		p = Pattern.Arbno(compile(n.left));
	    break;
	case ASSIGN:
	    p = Pattern.Assign(compile(n.left),n.right.varValue());
	    break;
	case BAL:
	    p = (n.left == null?Pattern.Bal()
			       :Pattern.Bal(n.left.stringValue()));
	    break;
	case BREAK:
	    switch ((l=n.left).code) {
	    case _STRING: p = Pattern.Break(l.stringValue()); break;
	    case _VAR: p = Pattern.Break(l.varValue()); break;
	    }
	    break;
	case BREAKX:
	    switch ((l=n.left).code) {
	    case _STRING: p = Pattern.BreakX(l.stringValue()); break;
	    case _VAR: p = Pattern.BreakX(l.varValue()); break;
	    }
	    break;
	case CAT:
	    p = Pattern.Concat(compile(n.left),compile(n.right));
	    break;
	case ABORT:
	case CANCEL:
	    p = Pattern.Cancel();
	    break;
	case DEFER:
	    p = Pattern.Defer(n.left.varValue());
	    break;
	case FAIL:
	    p = Pattern.Fail();
	    break;
	case FENCE:
	    if(n.left == null) {
		p = Pattern.Fence();
	    } else {
		p = compile(n.left);
		p = Pattern.Fence(p);
	    }
	    break;
	case IASSIGN:
	    p = Pattern.IAssign(compile(n.left),n.right.varValue());
	    break;
	case LEN:
	    l = n.left;
	    p = (l.code == _INT?Pattern.Len(l.intValue())
			       :Pattern.Len(l.varValue()));
	    break;
	case NOTANY:
	    switch ((l=n.left).code) {
	    case _STRING: p = Pattern.NotAny(l.stringValue()); break;
	    case _VAR: p = Pattern.NotAny(l.varValue()); break;
	    }
	    break;
	case NSPAN:
	    switch ((l=n.left).code) {
	    case _STRING: p = Pattern.NSpan(l.stringValue()); break;
	    case _VAR: p = Pattern.NSpan(l.varValue()); break;
	    }
	    break;
	case POS:
	    l = n.left;
	    p = (l.code == _INT?Pattern.Pos(l.intValue())
			       :Pattern.Pos(l.varValue()));
	    break;
	case REPLACE:
	    p = Pattern.Replace(compile(n.left),n.right.varValue());
	    break;
	case REM:
	case REST:
	    p = Pattern.Rest();
	    break;
	case RPOS:
	    l = n.left;
	    p = (l.code == _INT?Pattern.RPos(l.intValue())
			       :Pattern.RPos(l.varValue()));
	    break;
	case RTAB:
	    l = n.left;
	    p = (l.code == _INT?Pattern.RTab(l.intValue())
			       :Pattern.RTab(l.varValue()));
	    break;
	case SETCUR:
	    p = Pattern.Setcur(n.left.varValue());
	    break;
	case SPAN:
	    switch ((l=n.left).code) {
	    case _STRING: p = Pattern.Span(l.stringValue()); break;
	    case _VAR: p = Pattern.Span(l.varValue()); break;
	    }
	    break;
	case SUCCEED:
	    p = Pattern.Succeed();
	    break;
	case TAB:
	    l = n.left;
	    p = (l.code == _INT?Pattern.Tab(l.intValue())
			       :Pattern.Tab(l.varValue()));
	    break;

	case _INT:
	    p = Pattern.StringPattern(Integer.toString(n.intValue()));
	    break;
	case _STRING:
	    s = n.stringValue();
	    if(s.length() == 1)
		p = Pattern.CharPattern(s.charAt(0));
	    else
		p = Pattern.StringPattern(s);
	    break;
	case _VAR:
	    p = Pattern.Defer(n.varValue());
	    break;

	default:
	    throw new Error("Compiler error: unexpected Node: "+n.code);
	}
	return p;
    }

    //////////////////////////////////////////////////
    // Compile a pattern into a long nested set of 
    // Pattern package calls.

    public void compileJ(Parsetree n, boolean stringOK)
	throws Error
    {
	switch (n.code) {
	case ANY:
	case ARB:
	case BAL:
	case BREAK:
	case BREAKX:
	case CANCEL:
	case DEFER:
	case FAIL:
	case LEN:
	case NOTANY:
	case NSPAN:
	case POS:
	case REM:
	case REST:
	case RPOS:
	case RTAB:
	case SETCUR:
	case SPAN:
	case SUCCEED:
	case TAB:
	    if(n.left != null)
		compileJ(n.left,true);
	    if(n.right != null) compileJ(n.right,true);
	    // assemble pieces
	    n.text = "Pattern."+testOperator(n.code)+LPAREN;
	    if(n.left != null) n.text += n.left.text;
	    if(n.right != null) n.text += (", "+n.right.text);
	    n.text += RPAREN;
	    break;

	// Fence may or may not have a pattern argument
	case FENCE:
	    n.text = "Pattern."+testOperator(n.code)+LPAREN;
	    // check for 0 arg case
	    if(n.left == null) {
		n.text += RPAREN;
	    } else {
		// check 1 arg case
	        if(n.left != null) compileJ(n.left,false);
		n.text += n.left.text;
		n.text += RPAREN;
	    }
	    break;

	// Special case the operators with nested
	// pattern arguments
	case ARBNO:
	{
	    n.text = "Pattern."+testOperator(n.code)+LPAREN;
	    // check 1 arg case
	    if(n.left != null) {
		compileJ(n.left,true);
		n.text += n.left.text;
	    }
	    n.text += RPAREN;
	}; break;
	    
	// Special case two operand case
	case ALT:
	case CAT:
	case ASSIGN:
	case IASSIGN:
	case REPLACE:
	{
	    n.text = "Pattern."+testOperator(n.code)+LPAREN;
	    if(n.left != null) {
		compileJ(n.left,false);
		n.text += n.left.text;
	    }
	    if(n.right != null) {
		compileJ(n.right,false);
		n.text += (","+n.right.text);
	    }
	    n.text += RPAREN;
	}; break;

	case _STRING:
	{
	    String s = (DQUOTE // for java, always use DQUOTE
		        + QuotedString.addEscapes(n.stringValue(),DQUOTE)
		        + DQUOTE);
	    if(stringOK)
		n.text = s;
	    else {
		if(s.length() == 1)
		    n.text = ("Pattern.CharPattern"+LPAREN+SQUOTE
		              +QuotedString.addEscapedChar(s.charAt(0))
		              +SQUOTE+RPAREN);
		else
		    n.text = ("Pattern.StringPattern"+LPAREN+s+RPAREN);
	    }
	}; break;

	case _INT:
	    n.text = Integer.toString(n.intValue());
	    break;

	case _VAR:
	    n.text = ("Variable.create("+DQUOTE
		      +addEscapes(n.varValue().Name)
		      +DQUOTE+")");
	    break;

	default:
	    throw new Error("Compiler error: unexpected Node: "+n.code);
	}
    }


    static String indent = "                         ";
    static String getIndent(int n)
    {
	while(n > indent.length()) indent = (indent+indent);
	return indent.substring(0,n);
    }

    // Insert indentation into a nested Java Pattern expression.
    // Assume:
    // 1. All string constants are escaped
    // 2. There is no whitespace outside of a string constant

    // constants
    static String delims = "\",()";
    static String sdelims = "\"\\"; // for parsing inside string constants
    static String[] breakops = new String[] {
	"Pattern.Concat", "Pattern.Alternate"
    };

    public String indent(String expr,int iindent,int incr)
    {
	String curindent = "";
	int depth = 0;
	Stack<Integer> istack = new Stack<Integer>();
	istack.push(0); // prime stack
	StringBuilder buf = new StringBuilder();
	StringTokenizer lex = new StringTokenizer(expr,delims,true);

	while(lex.hasMoreTokens()) {
	    String tok = getToken(lex);
	    if(tok.length()==1 && delims.indexOf(tok.charAt(0)) >= 0) {
		switch (tok.charAt(0)) {
		case ',':
		    buf.append(tok);
		    // see if we are at the same depth as the stack top
		    if(depth == istack.peek()) {
			// force newline and indent
			buf.append("\n"+getIndent(iindent));
			buf.append(curindent);
		    }
		    break;
		case '(': buf.append(tok); depth++; break;
		case ')':
		    // check if we are closing the stack top depth
		    if(depth == istack.peek()) {
			// if so, then add newline and indent
			curindent = curindent.substring(incr);
			buf.append("\n"+getIndent(iindent));
			buf.append(curindent);
			istack.pop();
		    }
		    buf.append(tok);
		    depth--;
		    break;
		case '"': buf.append(tok); passString(buf,lex); break;
		}
	    } else { // basically an identifier
		// pass the operator
		buf.append(tok);
		if(isBreakop(tok)) {
		    // pass the following LPAREN
		    tok = getToken(lex);
		    buf.append(tok);
		    // bump the depth
		    depth++;
		    // stack this depth
		    istack.push(depth);
		    // alter the current indent
		    curindent = getIndent(depth*incr);
		    // add a newline
		    buf.append("\n"+getIndent(iindent));
		    // add in the indent
		    buf.append(curindent);
		}		
	    }	    
	}
	return buf.toString();
    }

    private void passString(StringBuilder buf, StringTokenizer lex)
    {
	while(lex.hasMoreTokens()) {
	    String tok = lex.nextToken(sdelims);
	    buf.append(tok); // pass this token
	    if(tok.equals("\"")) return;
	    if(tok.equals("\\")) {	    
		// pass next token
		tok = lex.nextToken(sdelims);
		buf.append(tok);
	    }
	}
    }

    private boolean isBreakop(String op)
    {
	for(String bop : breakops) {if(op.equals(bop)) return true;}
	return false;
    }

    private String getToken(StringTokenizer lex)
    {
	String tok=lex.nextToken(delims);
	Debug.level(3).Println("token="+tok);
	return tok;
    }

    //////////////////////////////////////////////////

    protected Parsetree parsePat() throws Error
    {
Debug.level(3).println("Pat");
	Parsetree n = parseAlt();
Debug.level(3).println("Pat: "+n);
	return n;
    }

    protected Parsetree parseAlt() throws Error
    {
Debug.level(3).println("Alt");
	Parsetree n = parseCat();
	if(n == null || ttype != Token.BAR) {
Debug.level(3).println("Alt0: "+n);
	    return n;
	}
Debug.level(3).println("Alt.l: "+n);
	Parsetree onode = new Parsetree(Keyword.ALT);
        onode.left = n;
	advance(); // skip BAR
	n = parseAlt();
	if(n == null)
	    throw new Error("Expected ALT: saw: "+token);
        onode.right = n;
Debug.level(3).println("Alt.f: "+onode);
	return onode;
    }

    protected Parsetree parseCat() throws Error
    {
Debug.level(3).println("Cat");
	Parsetree n = parseReplace();
	if(n == null) {
Debug.level(3).println("Cat0: "+n);
	    return n;
	}
Debug.level(3).println("Cat.l: "+n);
	// Check for explicit concatenation
	if(isAltFollow(ttype)) {// check for alt following
	    // looks like an alt follows, so quit
Debug.level(3).println("Cat1: "+n);
	    return n;
	} // else should be implicit concat or possibly EOF or RPAREN
	if(ttype == Token.AND) advance(); // consume the concat symbol
	Parsetree nr = parseCat();
	if(nr != null) {
	    Parsetree onode = new Parsetree(Keyword.CAT);
	    onode.left = n;
	    onode.right = nr;
	    n = onode;
	}
Debug.level(3).println("Cat.f: "+n);
	return n;
    }

    protected Parsetree parseReplace() throws Error
    {
Debug.level(3).println("Replace");
	Parsetree n = parseAssign();
	if(n == null) {
Debug.level(3).println("Replace: "+n);
	    return n;
	}
Debug.level(3).println("Replace.l: "+n);
	for(;;) {
	    Parsetree onode;
	    if(ttype == Token.EQUAL) {
	        onode = new Parsetree(Keyword.REPLACE);
	    } else
	        break;
	    onode.left = n;
	    advance(); // skip EQUAL
	    if(ttype != Token.ID)
	        throw new Error("Expected VAR: saw: "+token);
	    onode.right = new Parsetree(Keyword._VAR,new Variable(token));
	    advance(); // skip ID
	    n = onode;
	}
Debug.level(3).println("Replace.f: "+n);
	return n;
    }

    protected Parsetree parseAssign() throws Error
    {
Debug.level(3).println("Assign");
	Parsetree n = parseElement();
	if(n == null) {
Debug.level(3).println("Assign: "+n);
	    return n;
        }
Debug.level(3).println("Assign.l: "+n);
	for(;;) {
	    Parsetree onode;
	    if(ttype == Token.DOT || ttype == Token.STARSTAR) {
		onode = new Parsetree(Keyword.ASSIGN);
	    } else if(ttype == Token.DOLLAR || ttype == Token.STAR) {
	        onode = new Parsetree(Keyword.IASSIGN);
	    } else
	        break;
	    onode.left = n;
	    advance(); // skip STAR or STARSTAR
	    if(ttype != Token.ID)
	        throw new Error("Expected VAR: saw: "+token);
	    onode.right = new Parsetree(Keyword._VAR,new Variable(token));
	    advance(); // skip ID
	    n = onode;
	}
Debug.level(3).println("Assign.f: "+n);
	return n;
    }

    protected Parsetree parseElement() throws Error
    {
Debug.level(3).println("Element");
	Parsetree node = null;
Debug.level(3).println("Element.T: ttype="+Token.tokenName(ttype));
	switch (ttype) {

	case Token.BAR:
	case Token.AND:
	    break; // let next level up handle this
	    
	case Token.LPAREN:
Debug.level(3).println("LPAREN.1: ttype="+Token.tokenName(ttype));
	    advance();
Debug.level(3).println("LPAREN.2: ttype="+Token.tokenName(ttype));
	    node = parsePat();
Debug.level(3).println("LPAREN.3: ttype="+Token.tokenName(ttype)+" node="+node);
	    if(ttype != Token.RPAREN)
		throw new Error("Expected RPAREN: saw: "+token);
	    advance();
Debug.level(3).println("LPAREN.4: ttype="+Token.tokenName(ttype));
	    break;	    

	case Token.PLUS:
	    node = new Parsetree(Keyword.DEFER);
	    advance();
	    if(ttype != Token.ID)
		throw new Error("Expected ID: saw: "+token);
	    node.left = new Parsetree(Keyword._VAR,new Variable(token));
	    advance();
	    break;

	case Token.ID: 
	    node = parseOp(); 
	    break;

	case Token.EQUAL:
if(false) {
	    node = new Parsetree(Keyword.REPLACE);
	    advance();
	    if(ttype != Token.ID)
		throw new Error("Expected ID: saw: "+token);
	    node.left = new Parsetree(Keyword._VAR,new Variable(token));
	    advance();
}
	    break;

	case Token.DOLLAR:
	case Token.STAR:
if(false) {
	    node = new Parsetree(Keyword.IASSIGN);
	    advance();
	    if(ttype != Token.ID)
		throw new Error("Expected ID: saw: "+token);
	    node.left = new Parsetree(Keyword._VAR,new Variable(token));
	    advance();
}
	    break;

	case Token.DOT:
	case Token.STARSTAR:
if(false) {
	    node = new Parsetree(Keyword.ASSIGN);
	    advance();
	    if(ttype != Token.ID)
		throw new Error("Expected ID: saw: "+token);
	    node.left = new Parsetree(Keyword._VAR,new Variable(token));
	    advance();
}
	    break;

	case Token.INT: // fall thru; treat like string
	case Token.STRING:
	    node = new Parsetree(Keyword._STRING,token);
	    advance();
	    break;

	default:
	    break;
        }; // switch

Debug.level(3).println("Element.f: "+node);
        return node;
    }

    protected Parsetree parseSVarg() throws Error
    {
	Parsetree n = null;
	if((n = parseSarg()) != null) return n;
	if((n = parseVarg()) != null) return n;
	return null;
    }

    protected Parsetree parseNVarg() throws Error
    {
	Parsetree n = null;
	if((n = parseNarg()) != null) return n;
	if((n = parseVarg()) != null) return n;
	return null;
    }

    protected Parsetree parseSarg() throws Error
    {
	if(ttype != Token.STRING) return null;
	Parsetree n = new Parsetree(Keyword._STRING,token);
	advance();
	return n;
    }

    protected Parsetree parseNarg() throws Error
    {
	if(ttype != Token.INT) return null;
	Parsetree n = new Parsetree(Keyword._INT,Integer.parseInt(token));
	advance();
	return n;
    }

    protected Parsetree parseVarg() throws Error
    {
	if(ttype == Token.PLUS) { // this is optional but if there => Var
	    advance();
	    if(ttype != Token.ID)
		throw new Error("Expected ID: saw: "+token);
	} else if(ttype != Token.ID) return null;
	Parsetree node = new Parsetree(Keyword._VAR,new Variable(token));
	advance();
	return node;
    }

    // Parse a simple pattern operator
    protected Parsetree parseOp() throws Error
    {
Debug.level(3).println("Op");
        Keyword kw = Keyword.testKeyword(token);
	if(kw == Keyword.UNKNOWN) {
	    // Treat a simple ID (not a keyword)
	    // as if it had an implicit "+" in front of it
	    Parsetree node = new Parsetree(Keyword.DEFER);
	    node.left = new Parsetree(Keyword._VAR,new Variable(token));
	    advance();
	    return node;
	}

	// Special case, use advance0 to not skip white space
	// => expected lparen will be immediately after the
	// keyword

	Parsetree node = new Parsetree(kw);

	switch (kw) {

	case ANY:
	    paramStart();
	    node.left = parseSVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case ARB:
	    advance();
	    break;

	case ARBNO:
	    advance0();
	    if(ttype != Token.LPAREN)
	        throw new Error("Expected LPAREN; found:"+token);
	    // leave left paren as curren token
	    node.left = parseElement();
	    break;

	case BAL:
	    advance(); 
	    break;

	case BREAK:
	    paramStart();
	    node.left = parseSVarg();
	    if(node.left == null)	    		    	    
	        throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case BREAKX:
	    paramStart();
	    node.left = parseSVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case CANCEL:
	    advance(); 
	    break;

	case DEFER: // unary +
	    paramStart();
	    node.left = parseVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case FAIL:
	    advance(); 
	    break;

	case FENCE: // two cases here FENCE vs FENCE(P)
	    advance0();
	    if(ttype == Token.LPAREN) { // FENCE(P) case
		// leave left paren as curren token
		node.left = parseElement();
	    } else { // FENCE case
		if(ttype == Token.WHITESPACE) advance();
	    }
	    break;		

	case LEN:
	    paramStart();
	    node.left = parseNVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case NOTANY:
	    paramStart();
	    node.left = parseSVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case NSPAN:
	    paramStart();
	    node.left = parseSVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case POS:
	    paramStart();
	    node.left = parseNVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case REM:
	case REST:
	    advance();
	    break;

	case RPOS:
	    paramStart();
	    node.left = parseNVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case RTAB:
	    paramStart();
	    node.left = parseNVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case SETCUR:
	    paramStart();
	    node.left = parseVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case SPAN:
	    paramStart();
	    node.left = parseSVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case SUCCEED:
	    advance();
	    break;

	case TAB:
	    paramStart();
	    node.left = parseNVarg();
	    if(node.left == null)	    		    	    
		throw new Error("Expected parameter; found:"+token);
	    paramEnd();
	    break;

	case ALT:
	case ASSIGN:
	case IASSIGN:
	case CAT:
	default: throw new Error("Unexpected pattern: "+lex.token());
	    
	}
Debug.level(3).println("Op.f: "+node.toString());
	return node;
    }

    protected void paramStart() throws Error
    {
	advance0(); 
	if(ttype != Token.LPAREN)
	    throw new Error("Expected LPAREN; found:"+token);
	advance();
    }

    protected void paramEnd() throws Error
    {
	if(ttype != Token.RPAREN)
	    throw new Error("Expected RPAREN; found:"+token);
	advance();
    }

    protected void advance() throws Error
    {
	// skip ws and comments
	do {advance1();} while(ttype == Token.COMMENT);
    }

    protected void advance1() throws Error
    {
	// skip ws but not comments
	do {advance0();} while(ttype == Token.WHITESPACE);
    }

    protected void advance0() throws Error
    {
	// skip nothing
	ttype = lex.nextToken();
	token = lex.token();
    }
}

//////////////////////////////////////////////////
// Auxilliary Classes

//////////////////////////////////////////////////
enum Keyword
{
    ABORT, ALT, ANY, ARB, ARBNO, ASSIGN, BAL,
    BREAK, BREAKX, CAT, CANCEL, DEFER, FAIL,
    FENCE, IASSIGN, LEN, NOTANY,
    NSPAN, POS, REM, REPLACE, REST, RPOS,
    RTAB, SETCUR, SPAN, SUCCEED, TAB,
    // Following are markers for simple constants and variables
    // for use in class Node
    _STRING, _INT, _VAR,
    // If all else fails
    UNKNOWN;

    static public final Keyword END = UNKNOWN;
    static public final int size = (1+END.ordinal());

    static private String[] names = new String[] {
	"abort", "|", "any", "arb", "arbno", "$", "bal",
	"break", "breakx", "&", "cancel", "+", "fail",
	"fence", ".", "len", "notany",
	"nspan", "pos", "rem", "=", "rest", "rpos",
	"rtab", "setcur", "span", "succeed", "tab",
	"_STRING", "_INT", "_VAR",
	"",
    };

    static private String[] opnames= new String[] {
	"Abort", "Alternate", "Any", "Arb", "Arbno", "assign", "Bal",
	"Break", "BreakX", "Concat", "Abort", "Defer", "Fail",
	"Fence", "IAssign", "Len", "NotAny",
	"NSpan", "Pos", "Rem", "Replace", "Rem", "RPos",
	"RTab", "Setcur", "Span", "Succeed", "Tab",
	null, null, null, null
    };

    static HashMap<String,Keyword> Keywords;
    static Keyword testKeyword(String s)
        {return(Keywords.containsKey(s)?Keywords.get(s.toLowerCase()):UNKNOWN);}
    static HashMap<Keyword,String> JavaOps;
    static String testOperator(Keyword kw)
        {return (JavaOps.containsKey(kw)?JavaOps.get(kw):null);}

    static
    {
	// Establish keyword toname map
	Keywords = new HashMap<String,Keyword>();
	for(Keyword k : Keyword.values())
	    Keywords.put(names[k.ordinal()],k);

	// Establish the Keyword to Java operator map
	JavaOps = new HashMap<Keyword,String>();
	for(Keyword k : Keyword.values()) {
	    if(opnames[k.ordinal()] != null) {
		JavaOps.put(k,opnames[k.ordinal()]);
	    }
	}
    }

    // Define the set of simple valued nodes
    static EnumSet<Keyword> Simplevalued
	= EnumSet.of(_STRING, _INT, _VAR);

}; // enum Keyword

// Because it includes all characters, we can't
// implement Token as an enum (can we?) so fall back to constants
abstract class Token
{
    // Single char tokens
    static final int EOL = ('\n');
    static final int BAR = ('|');
    static final int PLUS = ('+');
    static final int AND = ('&');
    static final int STAR = ('*');
    static final int MINUS = ('-');
    static final int SLASH = ('/');
    static final int EQUAL = ('=');
    static final int DOLLAR = ('$');
    static final int DOT = ('.');
    static final int LPAREN = QuotedString.LPAREN;
    static final int RPAREN = QuotedString.RPAREN;
    static final int BACKSLASH = QuotedString.ESCAPE;

    static final int NOTOKEN = (0);

    // Non single char tokens
    static final int EOF = (-1);
    static final int WHITESPACE = (-2);
    static final int COMMENT = (-3);
    static final int ID = (-4);
    static final int INT = (-5);
    static final int STRING = (-6);
    static final int STARSTAR = (-7);

    static public String tokenName(int ttype)
    {
	String name = null;
	if(ttype <= 0) {
	    switch (ttype) {
		case NOTOKEN: name = "NOTOKEN"; break;
		case EOF: name = "EOF"; break;
		case WHITESPACE: name = "WHITESPACE"; break;
		case COMMENT: name = "COMMENT"; break;
		case ID: name = "ID"; break;
		case INT: name = "INT"; break;
		case STRING: name = "STRING"; break;
		case STARSTAR: name = "STARSTAR"; break;
		default: name = "unknown"+LPAREN+ttype+RPAREN; break;
	    }
	} else if(ttype == EOL) {
	    name = "\\n";
	} else {
	    name = "'" + ((char)ttype) + "'";
	}
	return name;
    }
}


class Lexer
{
    char quoteChar = QuotedString.DQUOTE;
    StringBuilder token;
    boolean pushedback;
    int ttype;
    CharStream in;

    public Lexer(CharStream cs)
    {
	in = cs;
	token = new StringBuilder();
	ttype = Token.NOTOKEN;
	pushedback=false;
    }

    public String token() {return token.toString();}
    public int tokenType() {return ttype;}
    public void pushBack() {pushedback = true;}
    public void setQuote(char q) throws Error
    {
	if(q != QuotedString.SQUOTE && q != QuotedString.DQUOTE)
	    throw new Error("Compiler.setQuote: illegal argument ("+q+")");
	quoteChar=q;
    }

    public int nextToken() throws Error
    {
	int ttype = nextToken0();
	Debug.level(2).print("nextToken: "+Token.tokenName(ttype));
	if(ttype < 0 && ttype != Token.EOF) Debug.level(2).print(" token=|"+token+"|");
	Debug.level(2).println();
	return ttype;
    }

    protected int nextToken0() throws Error
    {
	if(pushedback) {
	    pushedback = false;
	    return ttype;
	}
	token.setLength(0);
	int ch = in.getch();
	// All whitespace is accumulated into a singl token
	if(isWhiteSpace(ch)) {
	    do {
	        token.append((char)ch); ch = in.getch();
	    } while(isWhiteSpace(ch));
	    in.pushback(ch);
	    return (ttype=Token.WHITESPACE);
	}
	if(isID(ch)) {
	    do {
	        token.append((char)ch); ch = in.getch();
	    } while(isID2(ch));
	    in.pushback(ch);
	    return (ttype = Token.ID);
	}
	if(ch == Token.MINUS) {
	    token.append((char)ch);
	    ch = in.getch();
	if(!isDigit(ch)) {
	    in.pushback(ch);
	    return (ttype = Token.MINUS);
	}
	// negative number
	do {token.append((char)ch); ch = in.getch();} while(isDigit(ch));
	    in.pushback(ch);
	    return (ttype = Token.INT);
	}
	if(isDigit(ch)) {
	    do {token.append((char)ch); ch = in.getch();} while(isDigit(ch));
	    in.pushback(ch);
	    return (ttype = Token.INT);
	}
	if(ch == Token.STAR) {
	    // check to see if this is "**" vs "*"
	    token.append((char)ch);
	    ch = in.getch();
	    if(ch != Token.STAR) {
		in.pushback(ch);
		ttype = Token.STAR;
	    } else {
		token.append((char)ch);
	        ttype = Token.STARSTAR;
	    }
	    return ttype;
	} // star
	if(ch == quoteChar) {
	    // leave off quotes
	    in.pushback(ch);
	    if(QuotedString.removeEscapes(in,token,quoteChar) == null)
	        throw new Error("Unterminated String");
	    return (ttype = Token.STRING);
	} // quoteChar
	if(ch == Token.SLASH) { //possible comment
	    token.append((char)ch);
	    ch = in.getch();
	    if(ch != Token.SLASH && ch != Token.STAR) {
	        in.pushback(ch);
	        return (ttype = Token.SLASH);
	    }
	    // we have a comment of some kind
	    token.append((char)ch);
	    if(ch == Token.SLASH) {
	        // sweep to end of line (or eof)
	        ch = in.getch();
	        while(ch != Token.EOL && ch != Token.EOF) {
		    token.append((char)ch);
		    ch = in.getch();
		}
		if(ch == Token.EOL) token.append((char)ch);
	    } else { // /*...*/ comment
	        for(;;) {
		    ch = in.getch();
		    if(ch == Token.EOF)
		        throw new Error("Unterminated comment: "+token);
		    if(ch != Token.STAR) {token.append((char)ch); continue;}
		    // possible end of comment
		    token.append((char)ch); // pass the star
		    ch = in.getch();
		    if(ch == Token.SLASH) { // done
		        token.append((char)ch);
		        break;
		    }
		    // reread this next char
		    in.pushback(ch);
		}
	    }
	    return (ttype = Token.COMMENT);
	}
	if(ch == Token.BACKSLASH) {
	    ch = QuotedString.getEscapedChar(in);
	    // fall thru
	}
	if(ch == Token.EOF) {
	    token.append("EOF");
	    return (ttype = ch);
	}
	// default case: just pass as a delimiter or whitespace
	token.append((char)ch);
	return (ttype = ch);
    }

    static public boolean isID(int ch)
    {
	return (
	  (ch >= 'a' && ch <= 'z')
	   || (ch >= 'A' && ch <= 'Z')
	   || (ch >= '\u00A0' && ch <= '\u00FF')
	   || (ch == '_')
	);
    }

    // ID2 characters can appear after the initial id char
    static public boolean isID2(int ch)
	{return (isID(ch) || isDigit(ch));}

    static public boolean isDigit(int ch)
	{return ((ch >= '0' && ch <= '9'));}

    static public boolean isOctalDigit(int ch)
	{return ((ch >= '0' && ch <= '7'));}

    static public boolean isHexDigit(int ch)
    {
	return (
	  (ch >= '0' && ch <= '9')
	   || (ch >= 'A' && ch <= 'F')
	   || (ch >= 'a' && ch <= 'f')
	);
    }

    static public boolean isWhiteSpace(int ch)
	{return ((ch >= '\u0000' && ch <= '\u0020'));}

    static public int charToHex(int ch)
    {
	if(ch >= 'A' && ch <= 'Z') return ((ch - 'A')+10);
	if(ch >= 'a' && ch <= 'z') return ((ch - 'a')+10);
	// assume digit
	return (ch - '0');
    }

    static public int charToOctal(int ch)
	{return (ch - '0');}

}; // Lexer
