package jpattern.test;

import jpattern.compiler.Compiler;

import jpattern.util.*;
import jpattern.pattern.*;

import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.BufferedReader;	
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Collection;
import java.util.ArrayList;
import java.io.FileReader;
import java.io.FileWriter;

/*
import java.util.Date;
import java.text.SimpleDateFormat;
import java.text.DateFormat;
*/

public class Driver
{
    // Following are static because they are referenced
    // in other classes
    static Parameters parms;

    static InputStreamReader reader;
    static OutputStreamWriter writer;
    static OutputStreamWriter errwriter;
    static BufferedReader stdin;
    static PrintWriter stdout;
    static PrintWriter stderr;

   // Misc constants
    static char DQUOTE = QuotedString.DQUOTE;

    static final String DFALT_PACKAGE = "jpattern.test";

    // Define defalt cmdline arguments
    static String[] defaultFormals = new String[] {
	"envfile=",
	"env*",
	"f=",
	"o=",
	"err=",
	"debug?",
	"debugn#",
	"test=",
	"subject=",
	"pattern=",
	"package="
    };

    //////////////////////////////////////////////////

    static public void main(String[] argv) throws Exception
    {
	Driver main = new Driver();
	main.initialize(argv);
	main.start();
	main.cleanup();
	System.exit(0);
    }

    //////////////////////////////////////////////////

    // should only be called from subclass main
    void initialize(String[] argv) throws Exception
    {
	// Load in order, with later overriding earlier
	// 1. this class defaults
	// 2. env file
	// 3. command-line args

	Parameters mainparms = new Parameters(); // case 1
	Parameters envparms = new Parameters(); // case 2
	Parameters cmdparms = new Parameters(); // case 3

	setDefaults(mainparms); // case 1: main class defaults
	ParseArgs.parse(argv,(HashMap)cmdparms,getFormals());// case 2
	// Use cmdparms to find any env or envfile parameters

	// load any  env args first
	ArrayList envargs = (ArrayList)cmdparms.get("env");
	if(envargs != null) {
	    for(int i=0;i<envargs.size();i++)
		envparms.load1((String)envargs.get(i),false);
	}
	// load the env file, if any
	String envf = cmdparms.getString("envfile");
	if(envf != null) {envparms.load(envf,false);}

	// Now merge all the parameter sets
	parms = new Parameters();
	parms.putAll(mainparms);
	parms.putAll(envparms);
	parms.putAll(cmdparms);

	// Dump the arguments if -v
	if(parms.getBoolean("v")) {
	    System.err.println("parms="+parms.toString());
	}

	// Initialize any static class values for this and other classes
	// (must be a better way to do this)
	int lev = parms.getInt("debugn",0);
	Debug.setDebugn(lev);
	if(parms.getBoolean("debug")) Debug.setDebug(true);

	Driver.reader = new InputStreamReader(System.in);
	Driver.writer = new OutputStreamWriter(System.out);
	Driver.errwriter = new OutputStreamWriter(System.err);

	String infile = parms.getEnv("stdin");
	if(infile == null) infile = parms.getEnv("f");
	if(infile != null) {
	    Driver.reader=new FileReader(infile);
	}

	String outfile = parms.getEnv("stdout");
	if(outfile == null) outfile = parms.getEnv("o");
	if(outfile != null) {
	    Driver.writer=new FileWriter(outfile);
	}

	String errfile = parms.getEnv("stderr");
	if(errfile == null) errfile = parms.getEnv("err");
	if(errfile != null) {
	    Driver.errwriter=new FileWriter(errfile);
	}

	Driver.stdin = new BufferedReader(Driver.reader);
	Driver.stdout = new PrintWriter(Driver.writer,true);
	Driver.stderr = new PrintWriter(Driver.errwriter,true);

    }

    //////////////////////////////////////////////////
    // Subclass overrides

    public void start() throws Exception
    {
	String pattern= parms.getString("pattern");
	String subject= parms.getString("subject");
	String testclass = parms.getString("test");
	String prefix = parms.getString("package",DFALT_PACKAGE);
	if(subject == null)
	    throw new Exception("-subject must be specified");
	if(pattern == null)
	    throw new Exception("-pattern must be specified");
	if(testclass == null)
	    throw new Exception("-test must be specified");
	String className = prefix + "." + testclass;
	processClass(className, pattern, subject);
	
    }

    void processClass(String clname, String pattern, String subject)
	throws Exception
    {    
	PatternBuilder pb = (PatternBuilder)Factory.newInstance(clname);
	VarMap vars = pb.buildPattern();
	stdout.println("Testclass: "+clname+"; Pattern Set:");
	for(Object key : vars.keySet()) {
	    Object p = vars.get(key);	    
	    if(p instanceof Pattern) {
		stdout.println(key+" = "+((Pattern)p).graphToString());
	    } else if(p instanceof String) {
		stdout.println(key+" = "
			+DQUOTE+QuotedString.addEscapes((String)p)+DQUOTE);
	    }
	}
	Object p = vars.get(pattern);
	if(!(p instanceof Pattern))
	    throw new Exception(pattern+"not a pattern name in class "+clname);
	stdout.println("match: "
			+DQUOTE+QuotedString.addEscapes(subject)+DQUOTE
			+" pattern: "+pattern);
	match(vars, subject,(Pattern)p);
    }

    void match(VarMap vars, String subject, Pattern p) throws Exception
    {
	Match matcher = new Match();
	MatchResult result = new MatchResult();
	boolean ok = matcher.Match(subject,p,result,vars);
	String s = result.Subject;
	if(ok) {
stdout.println("matchresult="+result);
	    stdout.print("succeed: ");
	    String prefix = s.substring(0,result.Start);
	    String middle = s.substring(result.Start,result.Stop);
	    String suffix = s.substring(result.Stop,s.length());
	    stdout.println(prefix+"^"+middle+"^"+suffix);
	} else {
	    stdout.println("fail.");
	}
	stdout.println("vars after: "+vars.prettyPrint());
    }

    VarMap loadVars() throws Exception
    {
	VarMap vars = new VarMap();
	ArrayList varargs = (ArrayList)parms.get("var");
	if(varargs != null) {
	    for(int i=0;i<varargs.size();i++) {
		// parse each assignment
		String pair = (String)varargs.get(i);
		if(pair.charAt(pair.length()-1) == '*') {
		    // make the var be a multiple assignment
		    String nm = pair.substring(0,pair.length()-1);
		    ArrayList val = new ArrayList();
		    vars.put(new Variable(nm),val);
		} else {
		    int eqdex = pair.indexOf("=");
		    if(eqdex < 0) {
		        stderr.println("bad -var: "+pair);
		        continue;
		    }
		    String nm = pair.substring(0,eqdex);
		    String val = pair.substring(eqdex+1,pair.length());
		    vars.put(new Variable(nm),val);
		}
	    }
	}
	return vars;
    }

    public void cleanup() throws Exception {};

    public String[] getFormals()
    {
	return Driver.defaultFormals;
    }

    public void setDefaults(Parameters parms)
    {
	ArrayList<String> list = new ArrayList<String>();
	list.add("");
	list.add("snobol");
	list.add("jpattern.util");
	list.add("jpattern.compiler");
	parms.put("packageprefixes",list);
    }

    //////////////////////////////////////////////////

    static void usage(String msg) {usage(msg,null);}
    static void usage(Exception e) {usage(e.toString(),e);}

    static void usage(String msg, Exception e)
    {
	System.err.println(msg);
	if(e != null)
	    e.printStackTrace(System.err);
	System.exit(1);
    }

    //////////////////////////////////////////////////
    // Misc. Common routines

    static public boolean contains(int[] indices, int i)
    {
	if(indices == null) return false;
	for(int si : indices) if(si == i) return true;
	return false;
    }

    static public boolean contains(String[] ss, String s)
    {
	if(ss == null) return false;
	for(String si : ss) if(s.equals(si)) return true;
	return false;
    }

    static public String[] union(String[] fs1, String[] fs2)
    {
	String[] tmp = new String[fs1.length+fs2.length];
	int last = 0;
	if(fs1 != null) {
	    System.arraycopy(fs1,0,tmp,0,fs1.length);
	    last = fs1.length;
	}
	if(fs2 != null) {
	    for(String s: fs2) {
	        if(contains(fs1,s)) continue;
	        tmp[last++] = s;
	    }
	}
	String[] result = new String[last];
	System.arraycopy(tmp,0,result,0,last);
	return result;
    }

    static public String[] minus(String[] fs1, String[] fs2)
    {
	if(fs1 == null) return null;
	if(fs2 == null) return fs1;
	String[] tmp = new String[fs1.length];
	int last = 0;
	for(String s: fs1) {
	        if(!contains(fs2,s)) continue;
	        tmp[last++] = s;
	}
	String[] result = new String[last];
	System.arraycopy(tmp,0,result,0,last);
	return result;
    }

}